<?php

namespace App\Filament\Resources\Membership\MembershipPlanResource\Pages;

use App\Filament\Resources\Membership\MembershipPlanResource;
use Filament\Resources\Pages\ViewRecord;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Group;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\Tabs;
use Filament\Infolists\Components\Tabs\Tab;
use Filament\Support\Enums\FontWeight;
use Filament\Support\Enums\IconPosition;
use Illuminate\Support\Number;

class ViewMembershipPlan extends ViewRecord
{
    protected static string $resource = MembershipPlanResource::class;

    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Group::make([
                    Section::make()
                        ->schema([
                            TextEntry::make('package_title')
                                ->label('Package Title')
                                ->weight(FontWeight::Bold)
                                ->size(TextEntry\TextEntrySize::Large),
                            TextEntry::make('membership_type')
                                ->badge()
                                ->color('primary')
                                ->icon('heroicon-m-bookmark')
                                ->iconPosition(IconPosition::Before),
                            TextEntry::make('package_description')
                                ->markdown()
                                ->columnSpanFull(),
                        ])->columns(2),
                ])->columnSpanFull(),

                Tabs::make('Plan Information')
                    ->tabs([
                        Tab::make('Basic Details')
                            ->icon('heroicon-m-information-circle')
                            ->schema([
                                Grid::make(3)
                                    ->schema([
                                        Section::make('Role & Status')
                                            ->schema([
                                                TextEntry::make('role.name')
                                                    ->label('Role')
                                                    ->icon('heroicon-m-user-circle'),
                                                IconEntry::make('is_active')
                                                    ->label('Active Status')
                                                    ->boolean()
                                                    ->color('success'),
                                                IconEntry::make('is_visible')
                                                    ->label('Visibility')
                                                    ->boolean()
                                                    ->color('warning'),
                                                IconEntry::make('is_featured')
                                                    ->label('Featured')
                                                    ->boolean()
                                                    ->color('info'),
                                                TextEntry::make('display_order')
                                                    ->label('Display Order')
                                                    ->icon('heroicon-m-arrows-up-down'),
                                            ])->columnSpan(1),

                                        Section::make('Pricing Information')
                                            ->schema([
                                                TextEntry::make('price_monthly')
                                                    ->label('Monthly')
                                                    ->money('USD')
                                                    ->icon('heroicon-m-currency-dollar')
                                                    ->color('success'),
                                                TextEntry::make('price_quarterly')
                                                    ->label('Quarterly')
                                                    ->money('USD')
                                                    ->icon('heroicon-m-currency-dollar'),
                                                TextEntry::make('price_half_yearly')
                                                    ->label('Half Yearly')
                                                    ->money('USD')
                                                    ->icon('heroicon-m-currency-dollar'),
                                                TextEntry::make('price_yearly')
                                                    ->label('Yearly')
                                                    ->money('USD')
                                                    ->icon('heroicon-m-currency-dollar'),
                                            ])->columnSpan(1),

                                        Section::make('Trial')
                                            ->schema([
                                                IconEntry::make('has_trial')
                                                    ->label('Trial Available')
                                                    ->boolean()
                                                    ->color('success'),
                                                TextEntry::make('trial_days')
                                                    ->label('Trial Period')
                                                    ->suffix(' days')
                                                    ->icon('heroicon-m-calendar')
                                                    ->visible(fn ($record) => $record->has_trial),
                                                
                                            ])->columnSpan(1),
                                    ]),
                            ]),

                        Tab::make('Plan Modules')
                            ->icon('heroicon-m-puzzle-piece')
                            ->badge(fn ($record) => $record->planModules->count())
                            ->schema([
                                Section::make('Available Modules')
                                    ->description('Modules and features included in this plan')
                                    ->schema([
                                        Grid::make(3)
                                            ->schema([
                                                ...collect($this->record->planModules)->map(fn ($module) => 
                                                    Section::make($module->module_name)
                                                        ->collapsible()
                                                        ->icon('heroicon-m-puzzle-piece')
                                                        ->schema([
                                                            IconEntry::make("module_allow_listing")
                                                                ->state($module->module_allow_listing)
                                                                ->label('Can Post')
                                                                ->boolean()
                                                                ->color(fn ($state) => $state ? 'success' : 'danger'),
                                                            TextEntry::make("module_listing_limit")
                                                                ->state($module->module_listing_limit)
                                                                ->label('Listing Limit')
                                                                ->icon('heroicon-m-list-bullet'),
                                                            TextEntry::make("module_allowed_images")
                                                                ->state($module->module_allowed_images)
                                                                ->label('Images Per Listing')
                                                                ->icon('heroicon-m-photo'),
                                                            TextEntry::make("module_extra_listing_price")
                                                                ->state($module->module_extra_listing_price)
                                                                ->label('Extra Listing Price')
                                                                ->money('USD')
                                                                ->icon('heroicon-m-currency-dollar'),
                                                        ])
                                                        ->columnSpan(1)
                                                )->toArray()
                                            ]),
                                    ]),
                            ]),

                        Tab::make('Subscriptions')
                            ->icon('heroicon-m-user-group')
                            ->badge(fn ($record) => $record->subscriptions->count())
                            ->schema([
                                Grid::make(3)
                                    ->schema([
                                        Section::make('Subscription Statistics')
                                            ->icon('heroicon-m-chart-bar')
                                            ->schema([
                                                TextEntry::make('subscriptions_count')
                                                    ->label('Total Subscriptions')
                                                    ->state(fn ($record) => $record->subscriptions->count())
                                                    ->badge()
                                                    ->color('info')
                                                    ->icon('heroicon-m-users'),
                                                TextEntry::make('active_subscriptions')
                                                    ->label('Active Subscriptions')
                                                    ->state(fn ($record) => $record->subscriptions->where('is_active', true)->count())
                                                    ->badge()
                                                    ->color('success')
                                                    ->icon('heroicon-m-check-circle'),
                                                TextEntry::make('cancelled_subscriptions')
                                                    ->label('Cancelled Subscriptions')
                                                    ->state(fn ($record) => $record->subscriptions->whereNotNull('cancelled_at')->count())
                                                    ->badge()
                                                    ->color('danger')
                                                    ->icon('heroicon-m-x-circle'),
                                                TextEntry::make('trial_subscriptions')
                                                    ->label('Trial Subscriptions')
                                                    ->state(fn ($record) => $record->subscriptions->whereNotNull('trial_ends_at')->where('trial_ends_at', '>', now())->count())
                                                    ->badge()
                                                    ->color('warning')
                                                    ->icon('heroicon-m-clock'),
                                            ])->columnSpan(1),

                                        Section::make('Revenue Statistics')
                                            ->icon('heroicon-m-banknotes')
                                            ->schema([
                                                TextEntry::make('monthly_subscribers')
                                                    ->label('Monthly Subscribers')
                                                    ->state(fn ($record) => $record->subscriptions->where('payment_term', 'monthly')->where('is_active', true)->count())
                                                    ->badge()
                                                    ->icon('heroicon-m-calendar'),
                                                TextEntry::make('quarterly_subscribers')
                                                    ->label('Quarterly Subscribers')
                                                    ->state(fn ($record) => $record->subscriptions->where('payment_term', 'quarterly')->where('is_active', true)->count())
                                                    ->badge()
                                                    ->icon('heroicon-m-calendar'),
                                                TextEntry::make('yearly_subscribers')
                                                    ->label('Yearly Subscribers')
                                                    ->state(fn ($record) => $record->subscriptions->where('payment_term', 'yearly')->where('is_active', true)->count())
                                                    ->badge()
                                                    ->icon('heroicon-m-calendar'),
                                            ])->columnSpan(1),

                                        Section::make('Auto-Renewal Status')
                                            ->icon('heroicon-m-arrow-path')
                                            ->schema([
                                                TextEntry::make('auto_renewal_enabled')
                                                    ->label('Auto-Renewal Enabled')
                                                    ->state(fn ($record) => $record->subscriptions->where('auto_renewal', true)->where('is_active', true)->count())
                                                    ->badge()
                                                    ->color('success')
                                                    ->icon('heroicon-m-arrow-path'),
                                                TextEntry::make('auto_renewal_disabled')
                                                    ->label('Auto-Renewal Disabled')
                                                    ->state(fn ($record) => $record->subscriptions->where('auto_renewal', false)->where('is_active', true)->count())
                                                    ->badge()
                                                    ->color('danger')
                                                    ->icon('heroicon-m-x-circle'),
                                            ])->columnSpan(1),
                                    ]),
                            ]),

                        Tab::make('User Features')
                            ->icon('heroicon-m-cog')
                            ->schema([
                                Section::make('Profile Features')
                                    ->description('Features available on user profiles')
                                    ->schema([
                                        Grid::make(3)
                                            ->schema([
                                                Group::make([
                                                    IconEntry::make('packageUserInfo.show_website_link')
                                                        ->label('Website Link')
                                                        ->boolean(),
                                                    IconEntry::make('packageUserInfo.show_phone')
                                                        ->label('Phone Number')
                                                        ->boolean(),
                                                    IconEntry::make('packageUserInfo.show_address')
                                                        ->label('Address')
                                                        ->boolean(),
                                                ])->columnSpan(1),

                                                Group::make([
                                                    IconEntry::make('packageUserInfo.show_social_media_links')
                                                        ->label('Social Media')
                                                        ->boolean(),
                                                    IconEntry::make('packageUserInfo.searchable')
                                                        ->label('Profile Searchable')
                                                        ->boolean(),
                                                    IconEntry::make('packageUserInfo.show_public_review')
                                                        ->label('Public Reviews')
                                                        ->boolean(),
                                                ])->columnSpan(1),

                                                Group::make([
                                                    IconEntry::make('packageUserInfo.show_contact_form')
                                                        ->label('Contact Form')
                                                        ->boolean(),
                                                ])->columnSpan(1),
                                            ]),
                                    ]),

                                Section::make('Gallery Features')
                                    ->schema([
                                        Grid::make(2)
                                            ->schema([
                                                Group::make([
                                                    IconEntry::make('packageUserInfo.show_image_gallery')
                                                        ->label('Image Gallery')
                                                        ->boolean(),
                                                    TextEntry::make('packageUserInfo.image_gallery_limit')
                                                        ->label('Image Limit')
                                                        ->suffix(' images')
                                                        ->visible(fn ($record) => $record->packageUserInfo?->show_image_gallery),
                                                ])->columnSpan(1),

                                                Group::make([
                                                    IconEntry::make('packageUserInfo.show_video_gallery')
                                                        ->label('Video Gallery')
                                                        ->boolean(),
                                                    TextEntry::make('packageUserInfo.video_gallery_limit')
                                                        ->label('Video Limit')
                                                        ->suffix(' videos')
                                                        ->visible(fn ($record) => $record->packageUserInfo?->show_video_gallery),
                                                ])->columnSpan(1),
                                            ]),
                                    ]),
                            ]),
                    ])->columnSpanFull(),
            ]);
    }
}
